<?php

namespace App\Http\Controllers\Inventory;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Model\Product;
use App\Model\Measure;

class ProductController extends Controller
{
  /**
   * Display a listing of the resource.
   *
   * @return \Illuminate\Http\Response
   */
  public function index(Request $request)
  {
    $rows = Product::orderBy('id', 'desc')->get();
    $total = Product::count();

    foreach ($rows as $row) {

      $row['username'] = \FormHelper::userName($row->user_id);

      if (!empty($row->created_at))
        $row->created_on = $row->created_at->format('d-m-Y');
      else
        $row->created_on = '';

      $editUrl = \URL::to('/product/edit/') . '/' . $row->id;
      $viewUrl = \URL::to('/product/show/') . '/' . $row->id;
      $deleteUrl = \URL::to('/product/destroy/') . '/' . $row->id;
      $trackUrl = \URL::to('/product/tracker/') . '/' . $row->id;

      $row->action = '<a title="Show Product" href="' . $viewUrl . '" class="btn btn-sm btn-primary"><i class="fas fa-info"></i></a><a style="margin-left:4px" title="Track Product" href="' . $trackUrl . '" class="btn btn-sm btn-success"><i class="fa fa-asterisk"></i></a><a style="margin-left:4px" title="Edit Product" href="' . $editUrl . '" class="btn btn-sm btn-warning"><i class="fas fa-edit"></i></a><a style="margin-left:4px" title="Delete Product" href="' . $deleteUrl . '" class="btn btn-sm btn-danger"><i class="fas fa-trash-alt"></i></a>';
    }

    $json_data = array(
      "draw" => intval($request->input('draw')),
      "recordsTotal" => $total,
      "recordsFiltered" => $total,
      "data" => $rows
    );

    echo json_encode($json_data);
  }

  public function view()
  {
    return view('inventory.product.view');
  }

  public function create()
  {
    $measurements = Measure::orderBy('id', 'desc')->get();
    return view('inventory.product.create', ['measurements' => $measurements]);
  }

  public function store(Request $request)
  {
    try {

      Product::create($request->all());
      $messageType = 1;
      $message = "Product created successfully !";

    } catch (\Illuminate\Database\QueryException $ex) {
      $messageType = 2;
      $message = "Product creation failed !";
      \Log::error('Customer creation failed: ' . $ex->getMessage());
    }

    return redirect(url("/product/view"))->with('messageType', $messageType)->with('message', $message);
  }

  public function show($id)
  {
    $product = Product::where('id', $id)->first();
    return view('inventory.product.show')
      ->with('product', $product);
  }

  public function edit($id)
  {
    $product = Product::where('id', $id)->first();
    $measurements = Measure::orderBy('id', 'desc')->get();

    return view('inventory.product.edit')
      ->with('product', $product)
      ->with('measurements', $measurements);
  }

  public function tracker($id)
  {
    $product = Product::where('id', $id)->first();
    $products = \DB::table('products')
      ->select('quotations.id', 'quotations.quotation_number', 'customers.name as customer_name', 'quotation_items.quantity', 'quotation_items.total')
      ->join('quotation_items', 'products.id', '=', 'quotation_items.product_id')
      ->join('quotations', 'quotation_items.quotation_id', '=', 'quotations.id')
      ->join('customers', 'quotations.customer_id', '=', 'customers.id')
      ->where('products.id', '=', $id)
      ->where('quotation_items.deleted_at', '=', null)
      ->get();

    return view('inventory.product.tracker')
      ->with('products', $products)
      ->with('product', $product);
  }

  public function update(Request $request, $id)
  {
    try {

      $product = Product::find($id);
      $product->update($request->all());

      $messageType = 1;
      $message = "Product " . $product->name . " details updated successfully !";

    } catch (\Illuminate\Database\QueryException $ex) {
      $messageType = 2;
      $message = "Product updation failed !";
      \Log::error('Product updation failed: ' . $ex->getMessage());
    }

    return redirect(url("/product/view"))->with('messageType', $messageType)->with('message', $message);
  }

  public function destroy($id)
  {
    try {
      $product = Product::find($id);
      $product->delete();

      $messageType = 1;
      $message = "Product " . $product->name . " details deleted successfully !";

    } catch (\Illuminate\Database\QueryException $ex) {
      $messageType = 2;
      $message = "Product deletion failed !";
    }

    return redirect(url("/product/view"))->with('messageType', $messageType)->with('message', $message);
  }

  public function product_search(Request $request)
  {
    $term = $request['term'];
    $queries = Product::where('code', 'LIKE', '%' . $term . '%')->take(5)->get();

    foreach ($queries as $key => $value) {
      $queries[$key]['value'] = $value->code;
    }

    return \Response::json($queries);
  }

}
