<?php

namespace App\Http\Controllers\Inventory;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Model\Customer;
use App\Model\Quotation;
use App\Model\QuotationItem;
use App\Model\Product;
use Barryvdh\DomPDF\Facade as PDF;
use Illuminate\Support\Collection;
use App\Model\Vat;

class QuotationController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $length = isset($request['length']) ? $request['length'] : 10;
        $start  = isset($request['start']) ?  $request['start'] : 0;
        $dir    = isset($request['order'][0]['dir']) ?  $request['order'][0]['dir'] : 'desc';

        $total = Quotation::count();

        $columns = $request['columns'];
        $collection = new Collection($columns);

        $searchableItems = $collection->filter(function ($item) {
            return $item['searchable'] === 'true' && ($item['search']['value'] !== null || trim($item['search']['value']) !== '');
        });

        // echo '<pre>';
        // print_r($searchableItems);
        // die();

        $user = \Auth::user();

        \DB::enableQueryLog();

        if ($searchableItems->isNotEmpty()) {
            
            $query = Quotation::query()
                ->leftJoin('customers', 'quotations.customer_id', '=', 'customers.id')
                ->leftJoin('users', 'quotations.user_id', '=', 'users.id');
                
            $searchColumns = [
                'formated_quotation_date' => 'quotations.quotation_date',
                'quotation_number' => 'quotations.quotation_number',
                'customer.name' => 'customers.name',
                'username'      => ['users.firstname', 'users.lastname'],
            ];
            
            $searchableItems->each(function ($item) use ($query, $searchColumns) {
                $data = $item['data'];
                $searchValue = $item['search']['value'] ?? null;

                if (!empty($searchValue) && isset($searchColumns[$data])) {
                    $columns = $searchColumns[$data];

                    $query->where(function ($query) use ($columns, $searchValue) {
                        if (is_array($columns)) {
                            foreach ($columns as $column) {
                                $query->orWhere($column, 'LIKE', '%' . $searchValue . '%');
                            }
                        } else {
                            $query->orWhere($columns, 'LIKE', '%' . $searchValue . '%');
                        }
                    });
                }
            });

            if ($user->isAdmin()) {
                $rows = $query->selectRaw('quotations.*')
                    ->with('user', 'customer')
                    ->orderBy('quotations.id', 'desc')
                    ->limit($length)
                    ->offset($start)
                    ->get();
            }
            else
            {
                $rows = $query->selectRaw('quotations.*')
                    ->with('user', 'customer')
                    ->where('user_id', $user->id)
                    ->orderBy('quotations.id', 'desc')
                    ->limit($length)
                    ->offset($start)
                    ->get();
            }

            \Log::error('Query: ' . json_encode(\DB::getQueryLog()));             
            $total = $query->count();
            
            
   
        } else  {

            if ($user->isAdmin()) {
                $rows = Quotation::with('user','customer')->orderBy('id' , 'desc')->limit($length)->offset($start)->get();                
                $total = Quotation::count();
            }
            else
            {
                $rows = Quotation::with('user','customer')
                        ->where('user_id', $user->id)
                        ->orderBy('id' , 'desc')->limit($length)->offset($start)->get();                
                $total = Quotation::where('user_id', $user->id)->count();
            }
        }
        
        foreach ($rows as $row) {

            $row->formated_quotation_date = $row->formated_expiry_date = $fdata = $row->formated_updated_at = '';

            if(!empty($row->updated_at))
                $row->formated_updated_at = date('d-m-Y',strtotime($row->updated_at));

            if(!empty($row->quotation_date))
                $row->formated_quotation_date = date('d-m-Y',strtotime($row->quotation_date));

            if($row->status == Quotation::STATUS_DRAFT) 
                $fdata = \FormHelper::statusBtn(Quotation::STATUS_DRAFT,'warning');
            else if($row->status == Quotation::STATUS_ACCEPTED) 
                $fdata = \FormHelper::statusBtn(Quotation::STATUS_ACCEPTED,'success');
            else 
                $fdata = \FormHelper::statusBtn(Quotation::STATUS_REJECTED,'danger');
            

            $row['status'] = $fdata;
            $row['username'] = \FormHelper::userName($row->user_id);
            
            $editUrl = \URL::to('/quotation/edit/').'/'.$row->id;
            $viewUrl = \URL::to('/quotation/show/') . '/' . $row->id;
            $printUrl = \URL::to('/quotation/print/') . '/' . $row->id;
            $deleteUrl =  \URL::to('/quotation/destroy/').'/'.$row->id;

            //<a title="Show Quotation" href="' . $viewUrl . '" class="btn btn-sm btn-primary"><i class="fas fa-info"></i></a>

            $row->action = '<a target="_blank" title="Print Quotation Details" href="'.$printUrl.'" class="btn btn-sm btn-warning"><i class="fas fa-print"></i></a><a style="margin-left:4px" title="Edit Quotation Details" href="'.$editUrl.'" class="btn btn-sm btn-warning"><i class="fas fa-edit"></i></a><a style="margin-left:4px" title="Delete Quotation Details" href="'.$deleteUrl.'" class="btn btn-sm btn-danger"><i class="fas fa-trash-alt"></i></a>';
        
        }

        $json_data = array(
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => $total,
            "recordsFiltered" => $total,
            "data"            => $rows
        );

        echo json_encode($json_data);
    }

    public function view()
    {
        return view('inventory.quotation.index');
    }

    public function create()
    {
        $customers = Customer::get();
        $taxes = Vat::orderBy('id', 'desc')->get();
        $quotationRefStartCount = \Config::get('formdata.quotationRefStartCount') + Quotation::count();
        $customers = json_encode($this->__prepareCustomerData($customers));
        $quoteValidity = \Config::get('formdata.quoteValidity');
        $quotationRef = "RT/".sprintf("%02d", \Auth::id())."/".$quotationRefStartCount."/". date('y');

        return view('inventory.quotation.create', ['customers' => $customers, 'quoteValidity' => $quoteValidity, 'quotationRef' => $quotationRef, 'taxes' => $taxes]);
    }

    private function __prepareCustomerData($customers) {

    	$lcustomers = [];

        foreach ($customers as $customer) {

            $lcustomer['id']  = $customer->id;
            $lcustomer['text'] = $customer->name. ' ('.$customer->email.')';

            $lcustomers[] = $lcustomer;
        }

        return $lcustomers;
    }

    public function store(Request $request)
    {       
        try 
        {           
            $validatedData = $request->validate([
            //  'quotation_number' => 'required|string',
                'customer_id' => 'required|integer',
                'requested_by' => 'required|string',
                'quotation_date' => 'required|date_format:d-m-Y',
                'quotation_ref' => 'required|string',
                'enquiry_ref' => 'required|string',
                'notes' => 'nullable|string'
            ]);
            $input = $request->all();  

            //dd($input);

            $quotationCnt  = Quotation::withTrashed()->count();
            $quotation_number     = \FormHelper::generatingBillingId($quotationCnt);

            \DB::transaction(function() use ($input, $quotation_number) {

                $quotation                      = new Quotation();
                $quotation->user_id             = \Auth::id();
                $quotation->customer_id         = $input['customer_id'];
                $quotation->quotation_ref       = $input['quotation_ref'];
                $quotation->enquiry_ref         = $input['enquiry_ref'];
                $quotation->requested_by        = $input['requested_by'];
                $quotation->quotation_number    = $quotation_number;    
                $quotation->quotation_date      = date('Y-m-d',strtotime($input['quotation_date']));  
                $quotation->expiry_date         = date('Y-m-d');     
                $quotation->expiry_days         = 0; 
                $quotation->deliveryTerms       = $input['deliveryTerms'];
                $quotation->paymentTerms        = $input['paymentTerms'];
                $quotation->quoteValidity       = $input['quoteValidity'];
                $quotation->notes               = $input['notes'];
                $quotation->status              = Quotation::STATUS_DRAFT;            

                $quotation->sub_total           = $input['final_total_bill'];
                $quotation->total               = $input['grand_total'];
                $quotation->tax                 = $input['tax_amount'];
                $quotation->taxpercent       = $input['tax_percentage'];
                $quotation->save();

                $quotation_product = array();

                foreach ($input['item_name'] as $key => $value) {

                    $itemData = Product::where("id",$input['item_id'][$key])->first();
                    
                    $quotation_product[$key]['quotation_id'] = $quotation->id;
                    $quotation_product[$key]['name'] = $itemData->name;
                    $quotation_product[$key]['description'] = $itemData->description;
                    $quotation_product[$key]['unit'] = $input['unit'][$key];
                    $quotation_product[$key]['quantity'] = $input['item_quantity'][$key];
                    $quotation_product[$key]['price'] = $input['price'][$key];
                    $quotation_product[$key]['total'] = $input['sub_total'][$key];
                    $quotation_product[$key]['product_id'] = $input['item_id'][$key];
                }

                QuotationItem::insert($quotation_product);
            });

            $messageType = 1;
            $message = "Quotation created successfully !";
        } 
        catch(\Illuminate\Database\QueryException $ex){  
            $messageType = 2;
            $message = "Quotation creation failed !";                
            \Log::error('Quotation creation failed: ' . $ex->getMessage());        
        }

        return redirect(url("/quotation/view"))->with('messageType',$messageType)->with('message',$message);
    }

    public function show(Request $request, $id)
    {
        $quotation       = Quotation::with('customer','user','comments')->find($id);
        $quotationList   = QuotationItem::where('quotation_id', $id)->get();

        //dd($quotation); 

       // $pdf = Pdf::loadView('inventory.quotation.show', ['quotation' => $quotation, 'quotationList' => $quotationList]);
       // return $pdf->download('invoice.pdf');

        return view('inventory.quotation.show', ['quotation' => $quotation, 'quotationList' => $quotationList]);  
        // Pdf::view('inventory.quotation.show', ['quotation' => $quotation, 'quotationList' => $quotationList])
        //     ->format('a4')
        //     ->save('invoice.pdf');
    }

    public function print($id)
    {
        $quotation       = Quotation::with('customer','user')->find($id);
        $quotationList   = QuotationItem::with('product')->where('quotation_id', $id)->get();

        return view('inventory.quotation.print', ['quotation' => $quotation, 'quotationList' => $quotationList]);        
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $quotation       = Quotation::with('customer','user')->find($id);
        $quotationList   = QuotationItem::with('product')->where('quotation_id', $id)->get();
        $taxes = Vat::orderBy('id', 'desc')->get();

        $customers = Customer::get();
        $customers = json_encode($this->__prepareCustomerData($customers));
        $quoteValidity = \Config::get('formdata.quoteValidity');


        $statusArr = [Quotation::STATUS_DRAFT,Quotation::STATUS_ACCEPTED,Quotation::STATUS_REJECTED];
        
        return view('inventory.quotation.edit', ['statusArr' => $statusArr, 'customers' => $customers, 
        'quotation' => $quotation, 'quotationList' => $quotationList, 'quoteValidity' => $quoteValidity, 'taxes' => $taxes]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request)
    {      
        try 
        {    
            $input = $request->all();

            $quotation_id = $request['quotation_id'];

            \DB::transaction(function() use ($input, $quotation_id) {

                $quotation                      = Quotation::find($quotation_id);
                $quotation->user_id             = \Auth::id();
                $quotation->customer_id         = $input['customer_id'];
                $quotation->quotation_ref       = $input['quotation_ref'];
                $quotation->enquiry_ref         = $input['enquiry_ref'];
                $quotation->requested_by        = $input['requested_by'];
                $quotation->quotation_date      = date('Y-m-d',strtotime($input['quotation_date']));  
                $quotation->expiry_date         = date('Y-m-d');     
                $quotation->expiry_days         = 0; 
                $quotation->deliveryTerms       = $input['deliveryTerms'];
                $quotation->paymentTerms        = $input['paymentTerms'];
                $quotation->quoteValidity       = $input['quoteValidity'];
                $quotation->notes               = $input['notes'];
                $quotation->sub_total           = $input['final_total_bill'];
                $quotation->total               = $input['grand_total'];
                $quotation->tax                 = $input['tax_amount'];                
                $quotation->taxpercent          = $input['tax_percentage'];

                $quotation->update();

                $quotation_product = array();

                QuotationItem::where('quotation_id', $quotation_id)->delete();

                $quotation_product = array();

                foreach ($input['item_name'] as $key => $value) {

                    $itemData = Product::where("id",$input['item_id'][$key])->first();
                    
                    $quotation_product[$key]['quotation_id'] = $quotation_id;
                    $quotation_product[$key]['name'] = ""; //$itemData->name;
                    $quotation_product[$key]['description'] = ""; //$itemData->description;
                    $quotation_product[$key]['unit'] = $input['unit'][$key];
                    $quotation_product[$key]['quantity'] = $input['item_quantity'][$key];
                    $quotation_product[$key]['price'] = $input['price'][$key];
                    $quotation_product[$key]['total'] = $input['sub_total'][$key];
                    $quotation_product[$key]['product_id'] = $input['item_id'][$key];
                }

                QuotationItem::insert($quotation_product);
            });
        
            $messageType = 1;
            $message = " created successfully !";
            $message = "Quotation updated successfully !";

        } 
        catch(\Illuminate\Database\QueryException $ex){  
            $messageType = 2;
            $message = "Quotation update failed !";                
            \Log::error('Quotation update failed: ' . $ex->getMessage());        
        }        

        if($messageType == 1)
        	return redirect(url("/quotation/view"))->with(['success' => $message] );
       	else
       		return redirect(url("/quotation/view"))->with(['error' => $message] );
    }

    public function updateStatus(Request $request)
    {
        try 
        {    
            $input = $request->all();

            $quotation_id = $request['quotation_id'];

            \DB::transaction(function() use ($input, $quotation_id) {
                $quotation              = Quotation::find($quotation_id);
                $quotation->user_id     = \Auth::id();
                $quotation->status      = $input['status'];
                $quotation->update();
            });
        
            $messageType = 1;
            $message = " created successfully !";
            $message = "Quotation status updated successfully !";

        } 
        catch(\Illuminate\Database\QueryException $ex){  
            $messageType = 2;
            $message = "Quotation status update failed !";                
            \Log::error('Quotation status update failed: ' . $ex->getMessage());        
        }      
        
        if($messageType == 1)
        	return redirect(url("/quotation/view"))->with(['success' => $message] );
       	else
       		return redirect(url("/quotation/view"))->with(['error' => $message] );

    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($purchase_id)
    {
        try {
            
            $purchase = Quotation::find($purchase_id);
            $purchase->delete();    

            $messageType = 1;
            $message = "Quotation deleted successfully !";

        } catch(\Illuminate\Database\QueryException $ex){  
            $messageType = 2;
            $message = "Quotation deletion failed !";
        }
        
        return redirect(url("/quotation/view"))->with('messageType',$messageType)->with('message',$message);
    } 
}
