<?php

namespace App\Http\Controllers\Apps;

use App\DataTables\UsersDataTable;
use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Department;
use App\Models\UserSchool;
use App\Models\School;
use App\Models\Position;
use Illuminate\Http\Request;
use App\Services\BioTimeEmployeeService;
use Yajra\DataTables\Facades\DataTables;
use App\Jobs\SyncUser;
use Log;
use Carbon\Carbon;
use Illuminate\Support\Facades\Hash;
use Faker\Factory as Faker;


class UserManagementController extends Controller
{
    protected $bioTimeEmployeeService;

    public function __construct(BioTimeEmployeeService $bioTimeEmployeeService)
    {
        $this->bioTimeEmployeeService = $bioTimeEmployeeService;
    }
    /**
     * Display a listing of the resource.
     */
    public function index(UsersDataTable $dataTable)
    {
        return $dataTable->render('pages/apps.user-management.users.list');
    }

    public function syncEmployees()
    {

        //return "Work in progress";
        
        $emp_codes = ["161651","199460","200654","195060","601656","601657","173046","108779","160942","PASS194983","193581","201101","363137","221844","202068","PASS230117","Pass00221","161145"];

        $emails = ["Sajosh.Mathew@charterschools.ae","Susil.Karunathilake@charterschools.ae","Mouza.Alshamsi1@charterschools.ae","Washim.Khan@charterschools.ae","Ram.Gurung@charterschools.ae","Hashan.Gunawardena@charterschools.ae","Atharali.Agasimani@charterschools.ae","mohammed.imran@charterschools.ae","sgonsalve@aljanaen.ae","ADhasan@bloomcharterschools.ae","wawdalseed@bloomcharterschools.ae","slekha@bloomcharterschools.ae","iahmad@bloomcharterschools.ae","MPudiya@bloomcharterschools.ae","atharasingh@bloomcharterschools.ae","sthoduka@bloomcharterschools.ae","malshami@bloomcharterschools.ae","nnader@bloomcharterschools.ae"];


        $index = 0;

        foreach($emp_codes as $emp_code) {

            try {

                $areaData = $this->bioTimeEmployeeService->getEmployees($emp_code);
                
                if (is_null($areaData)) {           
                    Log::error('No data found. Please contact the system administrator.');
                    return;
                }
                
                foreach ($areaData as $key => $value) {
                    try {
                        $temp = [];
                        $faker = Faker::create();
                    
                        $empCode = trim($value['emp_code']);
                        $existingUser = User::where('username', $empCode)->first();
                    
                        if (!$existingUser) {
                            $temp['external_id'] = $value['id'];
                            $temp['name'] = trim($value['full_name']);
                            $email = trim($value['email']);
                    
                            if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
                                $temp['email'] = $emails[$index++]; //$faker->unique()->safeEmail;
                            } else {
                                $temp['email'] = $email;
                            }
                    
                            $temp['password'] = Hash::make('welcome');
                            $temp['first_name'] = trim($value['first_name']);
                            $temp['last_name'] = trim($value['last_name']);
                    
                            if (empty($empCode)) {
                                $temp['username'] = $faker->unique()->regexify('[A-Z0-9]{8}');
                            } else {
                                $temp['username'] = $empCode;
                            }
                    
                            // Ensure hire_date is a valid date format
                            if (!empty(trim($value['hire_date'])) && Carbon::createFromFormat('Y-m-d', trim($value['hire_date'])) !== false) {
                                $temp['hire_date'] = Carbon::createFromFormat('Y-m-d', trim($value['hire_date']))->toDateString();
                            } else {
                                $temp['hire_date'] = null; // Or handle invalid date format as needed
                            }
                    
                            // Assuming dept_code is available in $value['department'] and it's an array
                            $departmentCode = $value['department']['dept_code'] ?? null;
                            $department = Department::where('code', $departmentCode)->first();
                    
                            if ($department) {
                                $temp['department_id'] = $department->id;
                            }
                    
                            $newUser = User::create($temp);
                            User::find($newUser->id)->assignRole('employee');
                    
                            foreach ($value['area'] as $area) {
                                try {
                                    $school = School::where('code', $area['area_code'])->first();
                                    if ($school) {
                                        UserSchool::create([
                                            'user_id' => $newUser->id,
                                            'school_id' => $school->id,
                                        ]);
                                    }
                                } catch (\Exception $e) {
                                    Log::error('Failed to assign school to user. ' . $e->getMessage());
                                }
                            }        
                        }
                    } catch (\Exception $e) {
                        Log::error('Failed to process employee data. ' . $e->getMessage());
                    }
                }
            } catch (\Exception $e) {
                Log::error('Failed to retrieve employee data. ' . $e->getMessage());
            }
        }


        // try {
        //     SyncUser::dispatch($this->bioTimeEmployeeService);
        //     return redirect()->route('user-management.users.index')->with('success', 'Sync in progress. Please check back later to see the updated users.');

        // } catch (\Exception $e) {
        //    return redirect()->back()->with('error', 'There was an error initiating the sync process. Please try again later.');
        // }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function createEmployee()
    {       
        $departments = Department::all();
        $positions = Position::all();
        $schools = School::all();

        return view('pages/apps.user-management.users.createEmployee', compact('departments', 'positions', 'schools'));
    }

    public function editEmployee($user_id)
    {
        $user = User::with('schools')->find($user_id);
        $departments = Department::all();
        $positions = Position::all();
        $schools = School::all();

        $isLineManager = false;
        if($user->hasRole('Line Manager'))
            $isLineManager = true;

        return view('pages/apps.user-management.users.editEmployee', compact('user', 'departments', 'positions', 'schools', 'isLineManager'));
    }

    public function storeEmployee(Request $request)
    {
        $allInput = $request->all();
        $validated = $request->validate([
            'email' => 'required|email|max:191|unique:users,email',
            'first_name' => 'required|max:191',
            'last_name' => 'required|max:191',
            'hire_date' => 'nullable|date',
            'gender' => 'nullable|in:Male,Female,Other',
            'department_id' => 'nullable|exists:departments,id',
            'position_id' => 'nullable|exists:positions,id',
            'school_ids' => 'nullable|array',
            'school_ids.*' => 'exists:schools,id',
            'shift_id'  => 'nullable|exists:shift_info,id',
            'username' => 'required|max:191|unique:users,username',
            ], [
            'username.required' => 'The Employee Id field is required.',
            'username.max' => 'The Employee Id may not be greater than :max characters.',
            'username.unique' => 'The Employee Id has already been taken.',
        ]);

        $validated['name'] = $validated['first_name'].' '. $validated['last_name'];
        $validated['password'] = Hash::make(\Str::random(35));
        $validated['password_reset_token'] = \Str::random(60);
        $validated['password_reset_expires_at'] = now()->addHours(24);
        
        $user = User::create($validated);
        $user->assignRole('employee');

        UserSchool::where('user_id', $user->id)->delete();
        if($validated['school_ids'])
        {
            foreach ($validated['school_ids'] as $school_id) {
                UserSchool::create([
                    'user_id' => $user->id,
                    'school_id' => $school_id,
                ]);
            }
        }

        $isUserLineManager = false;

        if(isset($allInput['isLineManager']))
            $isUserLineManager = $allInput['isLineManager'] == "on" ? true : false;

        if($isUserLineManager)
        {
            $user->assignRole('Line Manager');
        }
        else 
        {
            if ($user->hasRole('Line Manager')) {
                $user->removeRole('Line Manager');
            }
        }
        return redirect()->route('user-management.users.index')->with('success', 'Employee inserted successfully!');
    }

    public function updateEmployee(Request $request, $userId)
    {
       // dd($request->all());
        \DB::beginTransaction();
        try { 

            $allInput = $request->all();
        
            $validated = $request->validate([
                'email' => 'required|email|max:191|unique:users,email,' . $userId,
                'first_name' => 'required|max:191',
                'last_name' => 'required|max:191',
                'hire_date' => 'nullable|date',
                'gender' => 'nullable|in:Male,Female,Other',
                'department_id' => 'nullable|exists:departments,id',
                'position_id' => 'nullable|exists:positions,id',
                'school_ids' => 'nullable|array',
                'school_ids.*' => 'exists:schools,id',
                'shift_id'  => 'nullable|exists:shift_info,id',
                'username' => 'required|max:191|unique:users,username,' . $userId,
                ], [
                    'username.required' => 'The Employee Id field is required.',
                    'username.max' => 'The Employee Id may not be greater than :max characters.',
                    'username.unique' => 'The Employee Id has already been taken.',
            ]);

            $validated['name'] = $validated['first_name'].' '. $validated['last_name'];
            $user = User::find($userId);
            $user->update($validated);

            $isUserLineManager = false;

            if(isset($allInput['isLineManager']))
                $isUserLineManager = $allInput['isLineManager'] == "on" ? true : false;

            if($isUserLineManager)
            {
                $user->assignRole('Line Manager');
            }
            else 
            {
                if ($user->hasRole('Line Manager')) {
                    $user->removeRole('Line Manager');
                }
            }

            UserSchool::where('user_id', $user->id)->delete();
            
            foreach ($validated['school_ids'] as $school_id) {
                UserSchool::create([
                    'user_id' => $user->id,
                    'school_id' => $school_id,
                ]);
            }      
            
            \DB::commit();

            return redirect()->route('user-management.users.index')->with('success', 'Employee updated successfully!');
        } catch (\Exception $e) {
            \DB::rollBack();
            \Log::error('Failed to Approve Attendance: '.$e->getMessage(), [
                'exception' => $e
            ]);
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     */
    public function show(User $user)
    {
        return view('pages/apps.user-management.users.show', compact('user'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(User $user)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, User $user)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(User $user)
    {
        //
    }

    public function isLineManager($user_id)
    {
        try {
            User::find($user_id)->assignRole('Line Manager');
        } catch(\Exception $e) {
            \Log::error('Failed to assignRole: '.$e->getMessage(), [
                'user_id' => $user_id,
                'exception' => $e
            ]);
            return redirect()->back()->with('error', 'Something wemt wrong!. Try again later.');
        }
        
        return redirect()->back()->with('success', "The employee's role has been updated to Line Manager.");
    }

    public function assignManager(Request $request, $user_id)
    {
        if ($request->ajax()) {
            // Fetch the initial collection
            $allLineManager = User::getUserWithDepartmentAndSchoolsByRole("Line Manager");

        // Apply the filter based on the search query
        if ($search = $request->input('search.value')) {
            $allLineManager = $allLineManager->filter(function ($item) use ($search) {
                return stripos($item->name, $search) !== false ||
                       stripos($item->email, $search) !== false ||
                       stripos(implode(', ', $item->school->pluck('name')->toArray()), $search) !== false;
            });
        }

        return DataTables::of($allLineManager)
            ->addIndexColumn()
            ->editColumn('school.name', function($row) {
                if ($row->schools->isNotEmpty()) {
                    return implode(', ', $row->schools->pluck('name')->toArray());
                } else {
                    return 'No Schools Assigned';
                }
            })
            ->addColumn('action', function($row)  use ($user_id) {
                $userData = User::find($user_id);
                if($userData->manager_id == $row->id) {
                    return '<button type="button"  class="btn btn-success btn-sm " disabled>Current Manager</button>';
                } else {
                    $assignUrl = route('user-management.assignManagerSubmit', ['user_id' => $user_id, 'manager_id' => $row->id]);
                    return '<a href="'.$assignUrl.'" class="btn btn-primary btn-sm">Assign</a>';
                }
            })
            ->rawColumns(['action'])
            ->make(true);
        }
        return view('pages/apps/user-management/users/assignManager', compact('user_id'));

    }

    public function assignManagerSubmit(Request $request, $user_id, $manager_id)
    {
        try {
            $user = User::find($user_id);
            if ($user) {
                $user->update(['manager_id' => $manager_id]);
            }
        } catch(\Exception $e) {
            \Log::error('Failed to assign manager: '.$e->getMessage(), [
                'user_id' => $user_id,
                'exception' => $e
            ]);
            return redirect()->back()->with('error', 'Something wemt wrong!. Try again later.');
        }
        
        return redirect()->back()->with('success', "The Line Manager has been assigned to the employee.");
    }
}
