<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\ShiftDetails;
use Carbon\Carbon;

class ShiftDetailsController extends Controller
{
    public function index($shift_info_id)
    {
        $shiftDetails = ShiftDetails::with('shiftInfo')->where('shift_info_id', $shift_info_id)->get();
        return view('pages/shift_manager/shiftdetails.index', compact('shift_info_id','shiftDetails'));
    }

    public function create($shift_info_id)
    {
        return view('pages/shift_manager/shiftdetails.create', ['shiftDetail' => null, 'shift_info_id' => $shift_info_id]);
    }

    public function store(Request $request)
    {

        try 
        {
            $request->validate([
                'shift_info_id' => 'required|exists:shift_info,id',
                'start_date' => 'required|date',
                'end_date' => 'nullable|date|after_or_equal:start_date',
                'day_of_week' => 'required|in:MF,MS,M,T,W,TH,F,S,Sun',
                'start_time' => 'required|date_format:H:i',
                'end_time' => 'required|date_format:H:i|after:start_time',
                'break_start_time' => 'nullable|date_format:H:i',
                'break_end_time' => 'nullable|date_format:H:i|after:break_start_time',
                'grace_in_minutes' => 'nullable|integer|min:0',
                'grace_out_minutes' => 'nullable|integer|min:0',
            ]);

            $data = $request->except('_token');
            if(isset($data['isDefault']))
                $data['isDefault'] = $data['isDefault'] == "on" ? true : false;
            else 
                $data['isDefault'] = false;

            $data['grace_out_minutes'] = $data['grace_out_minutes'] ?? 0;
            $data['grace_in_minutes'] = $data['grace_in_minutes'] ?? 0;

            $days = [];

            switch($data['day_of_week']) {
                case 'MF':
                    $days = ['M', 'T', 'W', 'TH', 'F'];
                    break;
                case 'MS':
                    $days = ['M', 'T', 'W', 'TH', 'F', 'S'];
                    break;
                default:
                    $days = [$data['day_of_week']];
                    break;
            }

            foreach ($days as $day) {
                $data['day_of_week'] = $day;
                ShiftDetails::create($data);
            }

            return redirect()->route('shift_details.index', $data['shift_info_id'])
                            ->with('success', 'Shift Details created successfully.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'An unexpected error occurred. Please try again.')->withInput();
        }
    }

    public function show(ShiftDetails $ShiftDetails)
    {
        return view('pages/shift_manager/shiftdetails.show', compact('ShiftDetails'));
    }

    public function edit($shift_details_id)
    {
        $shiftDetail = ShiftDetails::find($shift_details_id);
        return view('pages/shift_manager/shiftdetails.edit', compact('shiftDetail'));
    }

    private function covertToTimetoHiFormat($time)
    {
        $pattern = '/^(?:[01]\d|2[0-3]):(?:[0-5]\d):(?:[0-5]\d)$/';

        if (preg_match($pattern, $time)) {
            return Carbon::createFromFormat('H:i:s', $time)->format('H:i');
        } 

        return $time;
    }

    public function update(Request $request)
    {
        try 
        {
            //dd($request->all());

            $data = [
                'start_time' => $this->covertToTimetoHiFormat($request->input('start_time')),
                'end_time' => $this->covertToTimetoHiFormat($request->input('end_time')),
                'break_start_time' => $request->has('break_start_time') 
                    ? $this->covertToTimetoHiFormat($request->input('break_start_time'))
                    : null,
                'break_end_time' => $request->has('break_end_time') 
                    ? $this->covertToTimetoHiFormat($request->input('break_end_time'))
                    : null,
            ];
            
            $request->merge($data);            
            $request->validate([
                'shift_info_id' => 'required|exists:shift_info,id',
                'start_date' => 'required|date',
                'end_date' => 'nullable|date|after_or_equal:start_date',
                'day_of_week' => 'required|in:MF,MS,M,T,W,TH,F,S,Sun',
                'start_time' => 'required|date_format:H:i',
                'end_time' => 'required|date_format:H:i|after:start_time',
                'break_start_time' => 'nullable|date_format:H:i',
                'break_end_time' => 'nullable|date_format:H:i|after:break_start_time',
                'grace_in_minutes' => 'nullable|integer|min:0',
                'grace_out_minutes' => 'nullable|integer|min:0',
            ]);

            $data = $request->except('_token');
            if(isset($data['isDefault']))
                $data['isDefault'] = $data['isDefault'] == "on" ? true : false;
            else 
                $data['isDefault'] = false;           

            $data['grace_out_minutes'] = $data['grace_out_minutes'] ?? 0;
            $data['grace_in_minutes'] = $data['grace_in_minutes'] ?? 0;

            $shiftDetails = ShiftDetails::find($data['shift_details_id']);
            $shiftDetails->update($data);

            return redirect()->route('shift_details.index', $data['shift_info_id'])
                                ->with('success', 'Shift Details updated successfully.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'An unexpected error occurred. Please try again.')->withInput();
        }
    }

    public function destroy(ShiftDetails $ShiftDetails)
    {
        $ShiftDetails->delete();

        return redirect()->route('shift_masters.index')
                         ->with('success', 'Shift deleted successfully.');
    }
}

