<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\ShiftInfo;
use App\Models\School;
use Illuminate\Validation\Rule;


class ShiftManagerController extends Controller
{
    public function index()
    {
        $shifts = ShiftInfo::with('school')->get();
        return view('pages/shift_manager/shiftinfo.index', compact('shifts'));
    }

    public function create()
    {
        $schools = School::all();
        return view('pages/shift_manager/shiftinfo.create', compact('schools'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'user_id' => ['required', 'integer','exists:users,id'],
            'school_id' => ['required', 'integer', 'exists:schools,id'],
            'shift_code' => [
                'required', 'string', 'max:255', 
                    Rule::unique('shift_info')->where(function ($query) use ($request) {
                    return $query->where('school_id', $request->school_id);
                })
            ],
            'shift_name' => [ 'required', 'string','max:255'],
        ]);

        $data = $request->except('_token');
        ShiftInfo::create($data);

        return redirect()->route('shift_manager.index')
                         ->with('success', 'Shift has been created successfully.');
    }

    public function show(ShiftInfo $ShiftInfo)
    {
        return view('pages/shift_manager/shiftinfo.show', compact('ShiftInfo'));
    }

    public function edit($shiftInfo_id)
    {
        $shiftInfo = ShiftInfo::find($shiftInfo_id);
        $schools = School::all();
        return view('pages/shift_manager/shiftinfo.edit', compact('shiftInfo', 'schools'));
    }

    public function update(Request $request, $shift_info_id)
    {
        try 
        {
            $request->validate([
                'user_id' => ['required', 'integer','exists:users,id'],
                'school_id' => ['required', 'integer', 'exists:schools,id'],
                'shift_code' => [
                    'required', 'string', 'max:255', 
                        Rule::unique('shift_info')->where(function ($query) use ($request) {
                        return $query->where('school_id', $request->school_id);
                    })
                ],
                'shift_name' => [ 'required', 'string','max:255'],
            ]);

            $data = $request->except('_token');
            $shiftInfo = ShiftInfo::findOrFail($shift_info_id);
            $shiftInfo->update($data);

            return redirect()->route('shift_manager.index')
                            ->with('success', 'Shift updated successfully.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'An unexpected error occurred. Please try again.')->withInput();
        }
    }

    public function destroy(ShiftInfo $ShiftInfo)
    {
        $ShiftInfo->delete();

        return redirect()->route('shift_manager/shiftinfo.index')
                         ->with('success', 'Shift deleted successfully.');
    }

    public function postDefaultShiftData(Request $request)
    {
        $schoolIds = $request->input('school_ids');

        if (!$schoolIds || !is_array($schoolIds)) {
            return response()->json(['error' => 'No valid school IDs provided'], 400);
        }

        $schools = School::whereIn('id', $schoolIds)->get();
        
        if ($schools->isEmpty()) {
            return response()->json(['error' => 'No schools found'], 404);
        }
        
        $relatedData = ShiftInfo::whereIn('school_id', $schoolIds)->get();

        return response()->json([
            'schools' => $schools,
            'related_data' => $relatedData
        ]);
    }
}

