<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Carbon\Carbon;
use App\Models\TransactionSync;
use App\Services\BioTimeTransactionService;
use App\Models\TransactionRegularization;
use App\Models\Transaction;
use App\Models\User;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\Facades\DataTables;
use App\Notifications\AttendanceRequestSubmitted;
use App\Notifications\AttendanceRequestStatusUpdated;
use Illuminate\Support\Facades\Notification;

class TransactionController extends Controller
{
    protected $bioTimeTransactionService;

    public function __construct(BioTimeTransactionService $bioTimeTransactionService)
    {
        $this->bioTimeTransactionService = $bioTimeTransactionService;
    }
    public function UpdateTransactionsTime($user_id)
    {        
        $user = User::where('id', $user_id)->first();
        $start_time = $this->checkAndGetTheLastPunchTime($user_id)->format('Y-m-d H:i:s');
        $end_time = Carbon::now()->format('Y-m-d H:i:s');
        $this->bioTimeTransactionService->UpdateTransactionsTime($user->username, $user_id, $start_time, $end_time);
        return redirect()->back()->with('success', 'Attendence data successfully synced');
    }   

    private function checkAndGetTheLastPunchTime($user_id)
    {
        $transactionSync = TransactionSync::where('user_id', $user_id)->first();        
        if ($transactionSync) {
            $lastPunchTime =  Carbon::parse($transactionSync->last_punch_time)->addHour();
        } else {
            $lastPunchTime = Carbon::now()->subDays(45);
        }
        
        return $lastPunchTime;
    }

    public function myAttendance(Request $request)
    {       
        $userId = \Auth::id();
        $monthType = $request->input('monthType', 'current');
    
        if ($monthType === 'previous') {
            $endDate = Carbon::now()->subMonth()->endOfMonth();
            $startDate = Carbon::now()->subMonth()->startOfMonth();
        } else { 
            $endDate = Carbon::now();
            $startDate = Carbon::now()->startOfMonth();
        }

        $attendance = $this->bioTimeTransactionService->getMonthlyEmployeeAttendance($userId, $startDate, $endDate);        
        return view('pages/transaction.myAttendance', ['attendance' => $attendance, 'monthType' => $monthType]);
    }

    public function myAttendanceSubmit(Request $request)
    {
        try {
            DB::beginTransaction();
            $userId = \Auth::id();
            $date = $request->input('attendance_date');
            

            $rules = [
                'attendance_date' => 'required|date',
                'regularization_type' => 'required',
                'punch_in' => [
                    'required_if:regularization_type,in_time,in_out_time',
                    'nullable',
                    'date_format:H:i:s',
                    function ($attribute, $value, $fail) {
                        if (request('punch_out') && $value > request('punch_out')) {
                            $fail('The punch in time must be less than or equal to the punch out time.');
                        }
                    },
                ],
                'punch_out' => 'required_if:regularization_type,out_time,in_out_time|nullable|date_format:H:i:s',
                'reason' => 'required_if:regularization_type,in_time,out_time,in_out_time||nullable|string|max:255',
            ];
            
            $messages = [
                'attendance_date.required' => 'The date is required.',
                'punch_in.date_format' => 'The punch in time must be a valid time.',
                'punch_out.date_format' => 'The punch out time must be a valid time.',
                'reason.required_if' => 'The reason is required when modifying times.',
            ];
        
            $validatedData = $request->validate($rules, $messages);
            $validatedData['user_id'] = $userId;
            $userData = User::find(\Auth::id());
            $manager_id = 1; //todo need to be rewritten

            if($userData->manager_id) 
                $manager_id = $userData->manager_id;

            $validatedData['approved_by'] = $manager_id;
            $validatedData['approved_at'] = now();

            $transactionData = null;

            if ($request->input('regularization_type') === 'in_time') {

                $transactionData = Transaction::whereDate('punch_time', $date)
                                    ->where('user_id', $userId)
                                    ->orderBy('punch_time', 'asc')
                                    ->first();

                if ($transactionData) {
                    $transactionData->update(['is_corrected' => true]);
                    $validatedData['transaction_id'] = $transactionData->id;
                }
        
            } elseif ($request->input('regularization_type') === 'out_time') {
                
                $transactionData = Transaction::whereDate('punch_time', $date)
                                    ->where('user_id', $userId)
                                    ->orderBy('punch_time', 'desc')
                                    ->first(); 

                if ($transactionData) {
                    $validatedData['transaction_id'] = $transactionData->id;
                    $transactionData->update(['is_corrected' => true]);
                }    
            } elseif ($request->input('regularization_type') === 'in_out_time') {

                $transactionData = Transaction::whereDate('punch_time', $date)
                                    ->where('user_id', $userId)
                                    ->first(); 

                if ($transactionData) {                    
                    $validatedData['transaction_id'] = $transactionData->id;
                    $transactionData = Transaction::whereDate('punch_time', $request->input('date'))
                                        ->where('user_id', $userId)
                                        ->update(['is_corrected' => true]);
                } else {
                    $transactionData = Transaction::create([
                        'punch_time' => Carbon::parse($request->input('attendance_date'))->format('Y-m-d 00:00:00'),
                        'user_id' => $userId,
                        'is_corrected' => true
                    ]);
                    $validatedData['transaction_id'] = $transactionData->id;
                }
            } elseif ($request->input('regularization_type') === 'approved_leave') {
                $transactionData = Transaction::create([
                    'punch_time' => Carbon::parse($request->input('attendance_date'))->format('Y-m-d 00:00:00'),
                    'user_id' => $userId,
                    'is_corrected' => true,
                    'is_approved_leave' => true
                ]);
                $validatedData['transaction_id'] = $transactionData->id;
            }            

            $validatedData['status'] = "Pending";
            $validatedData['rejection_reason'] = "";

            $newTransactionRegularization = TransactionRegularization::updateOrCreate(
                [
                    'user_id' => $validatedData['user_id'],
                    'transaction_id' => $validatedData['transaction_id'],
                    'attendance_date' => $validatedData['attendance_date'],
                ], 
                $validatedData
            );

            $manager = User::find($manager_id); 
            Notification::send($manager, new AttendanceRequestSubmitted($newTransactionRegularization));

            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Failed to regularize transaction: '.$e->getMessage(), [
                'user_id' => \Auth::id(),
                'attendance_date' => $request->input('attendance_date'),
                'exception' => $e
            ]);

            return redirect()->back()->with('error', $e->getMessage());
        }
        return redirect()->back()->with('success', 'Attendance successfully regularized!');
    }

    public function requestView()
    {
        $pendingRegularization = TransactionRegularization::getListByStatus("pending");
        return view('pages/transaction.requestView', ['pendingRegularization' => $pendingRegularization]);        
    }

    public function reportView(Request $request)
    {
        if ($request->ajax()) {
            $data = $this->bioTimeTransactionService->getEmployeeAttendance(); 
            return DataTables::of($data)
                ->addIndexColumn()
                ->make(true);
        } 
        return view('pages/transaction.reportView');        
    }

    public function regularizeAttendance()
    {
        $usersWithTransactions = User::where('manager_id', \Auth::id())
                                ->whereHas('transactionRegularizations', function ($query) {
                                    $query->where('status', 'Pending');
                                })
                                ->with(['transactionRegularizations' => function ($query) {
                                    $query->where('status', 'Pending');
                                }])
                                ->get();
        return view('pages/transaction.regularizeAttendance', ['usersWithTransactions' => $usersWithTransactions]);
    }

    public function submitApproveAttendance(Request $request)
    {
        try {            
            DB::beginTransaction();

            $request->validate([
                'status' => 'required',
                'transactionRegularizationId' => 'required|exists:transaction_regularizations,id',
                'rejection_reason' => 'required_if:status,Rejected',
            ]);

            $transactionRegularizationId = $request->input('transactionRegularizationId');
            $transactionRegularizationData = TransactionRegularization::where('id', $transactionRegularizationId)->first();
            
            $rejectionReason = $request->input('rejection_reason', '');

            if ($transactionRegularizationData) {
                $transactionRegularizationData->update(['status' => $request->input('status'), 'rejection_reason' => $rejectionReason]);
            
                $employee = User::find($transactionRegularizationData->user->id); 
                Notification::send($employee, new AttendanceRequestStatusUpdated($transactionRegularizationData));
            } 

           
            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Failed to Approve Attendance: '.$e->getMessage(), [
                'exception' => $e
            ]);
            return redirect()->back()->with('error', $e->getMessage());
        }

        return redirect()->back()->with('success', 'Attendance successfully updated!');
    }
}
