<?php

namespace App\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use App\Services\BioTimeEmployeeService;
use Log;
use Illuminate\Support\Facades\Hash;
use Carbon\Carbon;
use Faker\Factory as Faker;
use App\Models\User;
use App\Models\Department;
use App\Models\UserSchool;
use App\Models\School;

class SyncUser implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $bioTimeEmployeeService;

    public function __construct(BioTimeEmployeeService $bioTimeEmployeeService)
    {
        $this->bioTimeEmployeeService = $bioTimeEmployeeService;
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        try {

            $areaData = $this->bioTimeEmployeeService->getEmployeesold();
            if (is_null($areaData)) {           
                Log::error('No data found. Please contact the system administrator.');
                return;
            }
            
            foreach ($areaData as $key => $value) {
                try {
                    $temp = [];
                    $faker = Faker::create();
                
                    $empCode = trim($value['emp_code']);
                    $existingUser = User::where('username', $empCode)->first();
                
                    if (!$existingUser) {
                        $temp['external_id'] = $value['id'];
                        $temp['name'] = trim($value['full_name']);
                        $email = trim($value['email']);
                
                        if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
                            $temp['email'] = $faker->unique()->safeEmail;
                        } else {
                            $temp['email'] = $email;
                        }
                
                        $temp['password'] = Hash::make('welcome_'.$empCode);
                        $temp['first_name'] = trim($value['first_name']);
                        $temp['last_name'] = trim($value['last_name']);
                
                        if (empty($empCode)) {
                            $temp['username'] = $faker->unique()->regexify('[A-Z0-9]{8}');
                        } else {
                            $temp['username'] = $empCode;
                        }
                
                        // Ensure hire_date is a valid date format
                        if (!empty(trim($value['hire_date'])) && Carbon::createFromFormat('Y-m-d', trim($value['hire_date'])) !== false) {
                            $temp['hire_date'] = Carbon::createFromFormat('Y-m-d', trim($value['hire_date']))->toDateString();
                        } else {
                            $temp['hire_date'] = null; // Or handle invalid date format as needed
                        }
                
                        // Assuming dept_code is available in $value['department'] and it's an array
                        $departmentCode = $value['department']['dept_code'] ?? null;
                        $department = Department::where('code', $departmentCode)->first();
                
                        if ($department) {
                            $temp['department_id'] = $department->id;
                        }
                
                        $newUser = User::create($temp);
                        User::find($newUser->id)->assignRole('employee');
                
                        foreach ($value['area'] as $area) {
                            try {
                                $school = School::where('code', $area['area_code'])->first();
                                if ($school) {
                                    UserSchool::create([
                                        'user_id' => $newUser->id,
                                        'school_id' => $school->id,
                                    ]);
                                }
                            } catch (\Exception $e) {
                                Log::error('Failed to assign school to user. ' . $e->getMessage());
                            }
                        }        
                    }
                } catch (\Exception $e) {
                    Log::error('Failed to process employee data. ' . $e->getMessage());
                }
            }
        } catch (\Exception $e) {
            Log::error('Failed to retrieve employee data. ' . $e->getMessage());
        }
    }
}
