<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use App\Models\User;

class AttendanceRequestStatusUpdated extends Notification
{
    
    use Queueable;

    private $transactionRegularization;

    public function __construct($transactionRegularization)
    {
        $this->transactionRegularization = $transactionRegularization;
    }

    public function via($notifiable)
    {
        //'mail',
        return ['database'];
    }

    public function toMail($notifiable)
    {        
        // Format the subject and body of the email
        $subject = 'Attendance Request has been ' . $this->transactionRegularization->status;
        $regularizationType = $this->transactionRegularization->regularization_type;
        
        // Create a new MailMessage instance
        $mailMessage = (new MailMessage)
            ->subject($subject)
            ->greeting('Hello,')
            ->line('Attendance Request has been ' . $this->transactionRegularization->status)
            ->line('Attendance Date: ' . $this->transactionRegularization->attendance_date)
            ->line('Regularization Type: ' . $this->getRegularizationTypeDescription($regularizationType))
            ->line('Reason: ' . (empty($this->transactionRegularization->reason) ? "-" : $this->transactionRegularization->reason));

        // Add conditional details based on the regularization type
        if ($regularizationType === 'in_time' || $regularizationType === 'in_out_time') {
            $mailMessage->line('Modified In Time: ' . $this->transactionRegularization->punch_in);
        }

        if ($regularizationType === 'out_time' || $regularizationType === 'in_out_time') {
            $mailMessage->line('Modified Out Time: ' . $this->transactionRegularization->punch_out);
        }

        if ($regularizationType === 'approved_leave') {
            $mailMessage->line('Leave Requested: Yes');
        }

        return $mailMessage;
    }

    /**
     * Helper function to return a more readable regularization type.
     */
    protected function getRegularizationTypeDescription($regularizationType)
    {
        switch ($regularizationType) {
            case 'in_time':
                return 'Date Modify In Time';
            case 'out_time':
                return 'Date Modify Out Time';
            case 'in_out_time':
                return 'Date Modify In/Out Time';
            case 'approved_leave':
                return 'Request to mark as approved leave';
            default:
                return 'Unknown';
        }
    }

    public function toDatabase($notifiable)
    {
        return [
            'attendance_request_id' => $this->transactionRegularization->id,
            'user_id' => $this->transactionRegularization->user_id,
            'type' =>  'AttendanceRequestSubmitted',
            'title' => 'Attendance Request has been ' . $this->transactionRegularization->status,
            'description' => 'Attendance Date: ' . date('d-m-y', strtotime($this->transactionRegularization->attendance_date)) . 
                 ' Regularization Type: ' . $this->getRegularizationTypeDescription($this->transactionRegularization->regularization_type),
            'submitted_at' => now()->format('d-m H:i')
        ];
    }
}

