<?php
namespace App\Services;
use Illuminate\Support\Facades\Http;

class BioTimeApiService
{
    protected $cacheKey = 'bio_time_auth_token';
    protected $tokenExpiryBuffer = 60; // seconds
    protected $baseUri;

    public function __construct()
    {
        $this->baseUri = env('BIOTIME_API_URL');
    }

    public function get($endpoint, $params = [], $headers = [])
    {        
        if(empty($headers))
            $response = Http::get($this->baseUri . $endpoint, $params);
        else 
            $response = Http::withHeaders($headers)->get($this->baseUri . $endpoint, $params);

        //echo $this->baseUri . $endpoint . '?' . http_build_query($params);
        //dd($response->json());

        if ($response->successful()) {
            return $response->json();
        }

        return $this->handleError($response);
    }

    public function post($endpoint, $data = [], $headers = [])
    {
        if(empty($headers))
            $response = Http::post($this->baseUri . $endpoint, $data);
        else 
            $response = Http::withHeaders($headers)->post($this->baseUri . $endpoint, $data);

        if ($response->successful()) {
            return $response->json();
        }

        return $this->handleError($response);
    }

    public function authenticateUser()
    {
        $cachedToken = $this->getCachedToken();
        if ($cachedToken) {
            return $cachedToken;
        }

        $url = 'api-token-auth/';
        $data = [
            'username' => env('BIOTIME_USERNAME'),
            'password' => env('BIOTIME_PASSWORD').'"',
        ];

        $headers = [
            'Content-Type' => 'application/json',
        ];

        $response = $this->post($url, $data, $headers);

        // Set the expiration time to 24 hours (86400 seconds)
        $expiresIn = 30 * 86400;

        if (isset($response['token'])) {
            $this->cacheToken($response['token'], $expiresIn);
            return $response['token'];
        }

        // Handle authentication failure (e.g., return null or throw an exception)
        return null;
    }

    protected function getCachedToken()
    {
        // Retrieve the token and its expiration time from the cache
        $token = cache()->get($this->cacheKey);
        $expiresAt = cache()->get($this->cacheKey . '_expires_at');

        if ($token && $expiresAt) {
            // Check if the token is still valid
            if (time() < $expiresAt - $this->tokenExpiryBuffer) {
                return $token;
            }

            // Token has expired
            cache()->forget($this->cacheKey);
            cache()->forget($this->cacheKey . '_expires_at');
        }

        return null;
    }

    protected function cacheToken(string $token, int $expiresIn)
    {
        $expiresAt = time() + $expiresIn;

        // Cache the token and its expiration time
        cache()->put($this->cacheKey, $token, $expiresIn);
        cache()->put($this->cacheKey . '_expires_at', $expiresAt, $expiresIn);
    }

    protected function handleError($response)
    {
        throw new \Exception("HTTP request failed with status " . $response->status());
    }
}
