<?php

namespace App\Services;
use App\Services\BioTimeApiService;
use App\Models\Transaction;
use App\Models\TransactionSync;
use App\Models\User;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class BioTimeTransactionService {
    
    protected $bioTimeApiService;

    public function __construct(BioTimeApiService $bioTimeApiService)
    {
        $this->bioTimeApiService = $bioTimeApiService;
    }

    public function getTransactions($emp_code, $start_time, $end_time)
    {
        $url = 'iclock/api/transactions/';
        $params = [];
        $params['page_size'] = 50;
        $params['emp_code'] = $emp_code;
        $params['start_time'] = $start_time;
        $params['end_time'] = $end_time;
        
        $allData = [];

        try 
        {
            $headers = [
                'Content-Type' => 'application/json',
                'Authorization' => 'Token '.$this->bioTimeApiService->authenticateUser(),
            ];

            do {
                $response = $this->bioTimeApiService->get($url, $params, $headers);
                
                if (isset($response['data'])) {
                    $allData = array_merge($allData, $response['data']);
                }
                
                if (!empty($response['next'])) {                
                    $urlComponents = parse_url($response['next']);
                    parse_str($urlComponents['query'], $params);
                } else {
                    $params = null; 
                }

            } while ($params);
            
            return $allData;

        } catch (\Exception $e) {
            // Handle the exception (logging can be added here if needed)
            return null;
        }
    }

    public function UpdateTransactionsTime($emp_code, $userId, $start_time, $end_time)
    {
        $transData = $this->getTransactions($emp_code, $start_time, $end_time);

        if($transData)
        {           
        $createData = [];

            foreach ($transData as $key => $value) {

                $temp['external_id'] = $value['id'];
                $temp['user_id'] = $userId;
                $temp['emp'] = $value['emp'];
                $temp['emp_code'] = $value['emp_code'];
                $temp['punch_time'] = $value['punch_time'];
                $temp['punch_state'] = $value['punch_state'];
                $temp['punch_state_display'] = $value['punch_state_display'];
                $temp['verify_type'] = $value['verify_type'];
                $temp['verify_type_display'] = $value['verify_type_display'];
                $temp['area_alias'] = $value['area_alias'];
                $temp['terminal_sn'] = $value['terminal_sn'];
                $temp['upload_time'] = $value['upload_time'];
                
                $createData[] = $temp;
            }
            
            if (!empty($createData)) {
                Transaction::insert($createData);

                $latestPunch = Transaction::orderBy('punch_time', 'desc')->first();
                $transaction = TransactionSync::where('user_id', $userId)->first();

                if ($transaction) {
                    $transaction->last_punch_time = $latestPunch->punch_time;
                    $transaction->save();
                } else {
                    TransactionSync::create([
                        "user_id" => $userId,
                        "last_punch_time" => $latestPunch->punch_time
                    ]);
                }
            }
        }
    }

    private function mergeCollections($collection1, $collection2) {
        return $mergedCollection = $collection1->map(function ($item1) use ($collection2) {
            // Find the matching item in the second collection
            $matchingItem = $collection2->first(function ($item2) use ($item1) {
                return $item1->date === $item2->attendance_date;
            });
            
            if ($matchingItem) {
                return (object) array_merge((array) $item1, $matchingItem->toArray());
            }
            
            return $item1;
        });        
    }

    private function getPunchData($userId, $startDate, $endDate)
    {
        return DB::table('transactions as t')
            ->leftJoin('transaction_regularizations as tr', function ($join) use ($userId) {
                $join->on(DB::raw('DATE(t.punch_time)'), '=', 'tr.attendance_date')
                    ->where('tr.user_id', $userId);
            })
            ->select(
                DB::raw('DATE(t.punch_time) as date'),
                DB::raw('MIN(t.punch_time) as login_time'),
                DB::raw('MAX(t.punch_time) as logout_time'),
                DB::raw('MAX(t.is_corrected) as correction_made'),
                DB::raw('MAX(t.is_approved_leave) as is_approved_leave'),
                DB::raw('MAX(tr.regularization_type) as regularization_type'),
                DB::raw('MAX(tr.punch_in) as punch_in'),
                DB::raw('MAX(tr.punch_out) as punch_out'),
                DB::raw('MAX(tr.status) as status'),                  
                DB::raw('MAX(COALESCE(tr.rejection_reason, "")) as rejection_reason')
                
            )
            ->where('t.user_id', $userId)
            ->whereBetween('t.punch_time', [$startDate, $endDate])
            ->groupBy(DB::raw('DATE(t.punch_time)'))
            ->orderBy('date', 'desc')
            ->get()
            ->keyBy('date');
    }

    public function getTodaysPunchData($userIds)
    {
        // Get today's date
        $today = Carbon::today()->toDateString();

        // Get transactions for users
        $transactions = Transaction::with(['user'])
            ->selectRaw('user_id, DATE(punch_time) as date')
            ->selectRaw('MIN(punch_time) as login_time')
            ->selectRaw('MAX(punch_time) as logout_time')
            ->whereIn('user_id', $userIds)
            ->whereDate('punch_time', $today)
            ->groupBy(DB::raw('DATE(punch_time)'), 'user_id')
            ->get();

        // Get users without transactions
        $usersWithoutTransactions = User::whereIn('id', $userIds)
            ->whereNotIn('id', $transactions->pluck('user_id'))
            ->get()
            ->map(function ($user) use ($today) {
                return [
                    'user_id' => $user->id,
                    'date' => $today,
                    'login_time' => 'N/A',
                    'logout_time' => 'N/A',
                    'user' => $user,
                ];
            });

        // Merge transactions with users without transactions
        if ($transactions->isEmpty()) {
            $results = $usersWithoutTransactions;
        } else {
            $results = $transactions->map(function ($transaction) {
                return [
                    'user_id' => $transaction->user_id,
                    'date' => $transaction->date,
                    'login_time' => $transaction->login_time,
                    'logout_time' => $transaction->logout_time,
                    'user' => $transaction->user,
                ];
            })->merge($usersWithoutTransactions);
        }
        
        return $results;
    }

    private function decimalToHoursMinutes($decimal)
    {
        $hours = floor($decimal);
        $minutes = ($decimal - $hours) * 60;

        return sprintf('%02d:%02d', $hours, $minutes);
    }

    public function getMonthlyEmployeeAttendance($userId, $startDate, $endDate)
    {
        // Fetch data from the transactions table
        $punchData = $this->getPunchData($userId, $startDate, $endDate);
        if($punchData->isEmpty())
            return [];

        // Initialize an array to hold the results
        $attendance = [];
        $numberOfDays = abs($endDate->diffInDays($startDate)) + 1;

        // Loop through the dates for the current month
        for ($i = 0; $i < $numberOfDays; $i++) {
            $date = $endDate->copy()->subDays($i)->format('Y-m-d');
            $dayOfWeek = Carbon::parse($date)->format('l');
            $punchForDate = $punchData->get($date);
            
            $status = [];
            $rejection_reason = "";
            $attendance[$date] = [
                'login_time' => null,
                'logout_time' => null,
                'hours_worked' => 0,
                'status' => '',
                'rejection_reason' => $rejection_reason,
                'is_corrected' => 0,
                'is_approved_leave' => 0,
            ];

            if ($punchForDate) {
                $loginTime = Carbon::parse($punchForDate->login_time);
                $logoutTime = Carbon::parse($punchForDate->logout_time);

                // Calculate hours worked
                $hoursWorked = abs($logoutTime->diffInHours($loginTime));
                $isRegularizationInOutTime = false;

                if ($punchForDate->regularization_type) {
                    if ($punchForDate->regularization_type == "in_time") {
                        $attendance[$date]['change_login_in'] = $punchForDate->punch_in;
                    } elseif ($punchForDate->regularization_type == "out_time") {
                        $attendance[$date]['change_login_out'] = $punchForDate->punch_out;
                    } elseif ($punchForDate->regularization_type == "in_out_time") {
                        $isRegularizationInOutTime = true;
                        $attendance[$date]['change_login_in'] = $punchForDate->punch_in;
                        $attendance[$date]['change_login_out'] = $punchForDate->punch_out;
                    }

                    $rejection_reason = $punchForDate->rejection_reason;

                    $status[] = $punchForDate->status;
                } else {
                    if ($hoursWorked > 8) {
                        $status[] = 'Present';
                    } elseif ($dayOfWeek === 'Sunday') {
                        $status[] = 'Rest Day';
                    } else {
                        if ($loginTime->gt(Carbon::parse($date . ' 09:00:00'))) {
                            $status[] = 'Late Punch In';
                        }
                        if ($logoutTime->lt(Carbon::parse($date . ' 17:00:00'))) { // Assuming 5 PM as standard logout time
                            $status[] = 'Early Punch Out';
                        }
                    }
                }
                
                $login_time_format = $loginTime->format('H:i:s');
                $logout_time_format = $logoutTime->format('H:i:s');

                if($isRegularizationInOutTime)
                {
                    if ($punchForDate->status === "Approved") {
                        $loginDate = Carbon::createFromFormat('Y-m-d', $punchForDate->date)->format('Y-m-d');

                        $login_time_format = Carbon::createFromFormat('Y-m-d H:i:s', $loginDate . ' ' . $punchForDate->punch_in);
                        $logout_time_format = Carbon::createFromFormat('Y-m-d H:i:s', $loginDate . ' ' . $punchForDate->punch_out);
                    }
                    else 
                    {
                        $login_time_format = $logout_time_format = "N/A";
                    }
                }

                $attendance[$date] = array_merge($attendance[$date], [
                    'login_time' =>  $login_time_format,
                    'logout_time' => $logout_time_format,
                    'hours_worked' => $this->decimalToHoursMinutes($hoursWorked),
                    'status' => implode(', ', $status),
                    'rejection_reason' => $rejection_reason,
                    'is_corrected' => $punchForDate->correction_made,
                    'is_approved_leave' => $punchForDate->is_approved_leave
                ]);
            } else {
                $attendance[$date]['status'] = ($dayOfWeek === 'Sunday') ? 'Rest Day' : 'Absent';
            }
        }

        return $attendance;
    }
    public function getCurrentMonthEmployeeDashboardAttendance(array $userIds = [])  
    {
        if(!$userIds)
            $userIds[] = \Auth::id();

        $endDate = Carbon::now();
        $startDate = Carbon::now()->startOfMonth(); 

        // Initialize the overall attendance count for all users
        $overallAttendance = [];

        foreach ($userIds as $userId) {
            // Fetch data from the transactions table for each user
            $punchData = $this->getPunchData($userId, $startDate, $endDate); 

            $attendance_count = [
                'present_days' => 0,
                'absent_days' => 0,
                'missed_login_in' => 0,
                'missed_login_out' => 0,
                'missed_login_in_out' => 0,
                'user' => User::find($userId)
            ];

            if ($punchData->isEmpty()) {
                $overallAttendance[$userId] = $attendance_count;
                continue;
            }

            $punchDataByDate = $punchData->keyBy('date');
            $currentDate = $endDate->copy();

            while ($currentDate->greaterThanOrEqualTo($startDate)) {
                $date = $currentDate->format('Y-m-d');
                $dayOfWeek = $currentDate->format('l');
                $punchForDate = $punchDataByDate->get($date);

                if ($punchForDate) {
                    $loginTime = Carbon::parse($punchForDate->login_time);
                    $logoutTime = Carbon::parse($punchForDate->logout_time);
                    $hoursWorked = abs($logoutTime->diffInHours($loginTime));

                    if (isset($punchForDate->regularization_type)) {

                        if($punchForDate->status == "Approved") {
                            $attendance_count['present_days']++;
                        } else {
                            switch ($punchForDate->regularization_type) {
                                case 'in_time':
                                    $attendance_count['missed_login_in']++;
                                    break;
                                case 'out_time':
                                    $attendance_count['missed_login_out']++;
                                    break;
                                case 'in_out_time':
                                    $attendance_count['missed_login_in_out']++;
                                    break;
                                case 'approved_leave':
                                    $attendance_count['absent_days']++;
                                    break;
                            }
                        }
                    } else if ($hoursWorked > 8) {
                        $attendance_count['present_days']++;
                    } else if ($dayOfWeek !== 'Sunday') {
                        $morningTime = Carbon::parse($date . ' 09:00:00');
                        $eveningTime = Carbon::parse($date . ' 17:00:00');
                        
                        if ($loginTime->greaterThan($morningTime) && $logoutTime->lessThan($eveningTime)) {
                            $attendance_count['missed_login_in_out']++;
                        } else if ($loginTime->greaterThan($morningTime)) {
                            $attendance_count['missed_login_in']++;
                        } else if ($logoutTime->lessThan($eveningTime)) {
                            $attendance_count['missed_login_out']++;
                        }
                    }
                } else if ($dayOfWeek !== 'Sunday') {
                    $attendance_count['absent_days']++;
                }

                $currentDate->subDay();
            }

            // Store the attendance count for the current user
            $overallAttendance[$userId] = $attendance_count;
        }

        return $overallAttendance;
    }


    public function getEmployeeAttendance()
    {
        $userId = \Auth::id();
        $endDate = Carbon::now();
        $startDate = Carbon::now()->subDays(90)->startOfMonth();  

        $punchData = $this->getPunchData($userId, $startDate, $endDate);  
        if($punchData->isEmpty())
            return [];

        // Initialize an array to hold the results
        $attendance = [];
        $numberOfDays = 90; 

        // Loop through the dates for the last 30 days
        for ($i = 0; $i < $numberOfDays; $i++) {
            $date = $endDate->copy()->subDays($i)->format('Y-m-d');
            $dayOfWeek = Carbon::parse($date)->format('l');

            // Find the punch data for the specific date
            $punchForDate = $punchData->firstWhere('date', $date);

            if ($punchForDate) {
                $loginTime = Carbon::parse($punchForDate->login_time);
                $logoutTime = Carbon::parse($punchForDate->logout_time);
                
                $attendance[$date] = [
                    'date' => $date,
                    'login_time' => $loginTime->format('H:i:s'),
                    'logout_time' => $logoutTime->format('H:i:s'),                    
                    'is_corrected' => $punchForDate->correction_made,
                    'is_approved_leave' => $punchForDate->is_approved_leave
                ];
            } else {
                if ($dayOfWeek === 'Sunday') {
                    $attendance[$date] = [
                        'date' => $date,
                        'login_time' => "-",
                        'logout_time' => "-",
                        'is_corrected' => false,
                        'is_approved_leave' => false
                    ];
                } else {
                    $attendance[$date] = [
                        'date' => $date,
                        'login_time' => "AB",
                        'logout_time' => "AB",
                        'is_corrected' => false,
                        'is_approved_leave' => false
                    ];
                }
            }
        }
        
        return $attendance;
    }
    
}