<?php

namespace App\Http\Controllers\Inventory;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Model\Customer;
use App\Model\Quotation;
use App\Model\QuotationItem;
use App\Model\Product;

class QuotationController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $length = isset($request['length']) ? $request['length'] : 10;
        $start  = isset($request['start']) ?  $request['start'] : 0;
        $dir    = isset($request['order'][0]['dir']) ?  $request['order'][0]['dir'] : 'desc';

        $total = $filtered_total = Quotation::count();

        $search = $request['search']['value'];

        if (empty($search)) {

            $rows = Quotation::with('user','customer')->orderBy('id' , 'desc')->limit($length)->offset($start)->get();
            $total = Quotation::count();
        } else {

            $rows  = new Quotation();
            $rows  = $rows->with('user','customer')->where('estimate_number', 'LIKE', '%' . $search . '%');
            $total = Quotation::where('estimate_number', 'LIKE', '%' . $search . '%')->count();

            if($length != -1)
                $rows  = $rows->limit($length)->offset($start);

            $rows  = $rows->orderBy('id' , 'desc')->get();

        }

        foreach ($rows as $row) {

            $row->formated_created_at = $row->formated_quotation_date = $row->formated_expiry_date = $fdata = '';

            if(!empty($row->created_at))
                $row->formated_created_at = date('d-m-Y',strtotime($row->created_at));

            if(!empty($row->quotation_date))
                $row->formated_quotation_date = date('d-m-Y',strtotime($row->quotation_date));

            if(!empty($row->created_at))
                $row->created_on = date('d-m-Y',strtotime($row->created_at));
            else
                $row->created_on = '';

            if($row->status == Quotation::STATUS_DRAFT) 
                $fdata = \FormHelper::statusBtn(Quotation::STATUS_DRAFT,'warning');
            else if($row->status == Quotation::STATUS_ACCEPTED) 
                $fdata = \FormHelper::statusBtn(Quotation::STATUS_ACCEPTED,'success');
            else 
                $fdata = \FormHelper::statusBtn(Quotation::STATUS_REJECTED,'danger');
            

            $row['status'] = $fdata;
            
            $editUrl = \URL::to('/quotation/edit/').'/'.$row->id;
            $viewUrl = \URL::to('/quotation/show/') . '/' . $row->id;
            $printUrl = \URL::to('/quotation/print/') . '/' . $row->id;
            $deleteUrl =  \URL::to('/quotation/destroy/').'/'.$row->id;

            $row->action = '<a title="Show Quotation" href="' . $viewUrl . '" class="btn btn-sm btn-primary"><i class="fas fa-info"></i></a><a style="margin-left:4px" target="_blank" title="Print Quotation Details" href="'.$printUrl.'" class="btn btn-sm btn-warning"><i class="fas fa-print"></i></a><a style="margin-left:4px" title="Edit Quotation Details" href="'.$editUrl.'" class="btn btn-sm btn-warning"><i class="fas fa-edit"></i></a><a style="margin-left:4px" title="Delete Quotation Details" href="'.$deleteUrl.'" class="btn btn-sm btn-danger"><i class="fas fa-trash-alt"></i></a>';
        
        }

        $json_data = array(
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => $total,
            "recordsFiltered" => $total,
            "data"            => $rows
        );

        echo json_encode($json_data);
    }

    public function view()
    {
        return view('inventory.quotation.index');
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $customers = Customer::get();
        $customers = json_encode($this->__prepareCustomerData($customers));

        return view('inventory.quotation.create', ['customers' => $customers]);
    }

    private function __prepareCustomerData($customers) {

    	$lcustomers = [];

        foreach ($customers as $customer) {

            $lcustomer['id']  = $customer->id;
            $lcustomer['text'] = $customer->name. ' ('.$customer->email.')';

            $lcustomers[] = $lcustomer;
        }

        return $lcustomers;
    }

    public function store(Request $request)
    {       
        try 
        {           
            $validatedData = $request->validate([
            //  'quotation_number' => 'required|string',
                'customer_id' => 'required|integer',
                'requested_by' => 'required|string',
                'quotation_date' => 'required|date_format:d-m-Y',
                'quotation_ref' => 'required|string',
                'enquiry_ref' => 'required|string',
                'notes' => 'nullable|string'
            ]);
            $input = $request->all();  

            //dd($input);

            $quotationCnt  = Quotation::withTrashed()->count();
            $quotation_number     = \FormHelper::generatingBillingId($quotationCnt);

            \DB::transaction(function() use ($input, $quotation_number) {

                $quotation                      = new Quotation();
                $quotation->user_id             = \Auth::id();
                $quotation->customer_id         = $input['customer_id'];
                $quotation->quotation_ref       = $input['quotation_ref'];
                $quotation->enquiry_ref         = $input['enquiry_ref'];
                $quotation->requested_by        = $input['requested_by'];
                $quotation->quotation_number    = $quotation_number;    
                $quotation->quotation_date      = date('Y-m-d',strtotime($input['quotation_date']));  
                $quotation->expiry_date         = date('Y-m-d');     
                $quotation->expiry_days         = 0; 
                $quotation->notes               = $input['notes'];
                $quotation->status              = Quotation::STATUS_DRAFT;            

                $quotation->sub_total           = $input['final_total_bill'];
                $quotation->total               = $input['grand_total'];
                $quotation->tax                 = $input['tax_percent'];
                $quotation->save();

                $quotation_product = array();

                foreach ($input['item_name'] as $key => $value) {

                    $itemData = Product::where("id",$input['item_id'][$key])->first();
                    
                    $quotation_product[$key]['quotation_id'] = $quotation->id;
                    $quotation_product[$key]['name'] = $itemData->name;
                    $quotation_product[$key]['description'] = $itemData->description;
                    $quotation_product[$key]['unit'] = $input['unit'][$key];
                    $quotation_product[$key]['quantity'] = $input['item_quantity'][$key];
                    $quotation_product[$key]['price'] = $input['price'][$key];
                    $quotation_product[$key]['total'] = $input['sub_total'][$key];
                    $quotation_product[$key]['product_id'] = $input['item_id'][$key];
                }

                QuotationItem::insert($quotation_product);
            });

            $messageType = 1;
            $message = "Quotation created successfully !";
        } 
        catch(\Illuminate\Database\QueryException $ex){  
            $messageType = 2;
            $message = "Quotation creation failed !";                
            \Log::error('Quotation creation failed: ' . $ex->getMessage());        
        }

        return redirect(url("/quotation/view"))->with('messageType',$messageType)->with('message',$message);
    }

    public function show($id)
    {
        $quotation       = Quotation::with('customer','user','comments')->find($id);
        $quotationList   = QuotationItem::where('quotation_id', $id)->get();

        return view('inventory.quotation.show', ['quotation' => $quotation, 'quotationList' => $quotationList]);        
    }

    public function print($id)
    {
        $quotation       = Quotation::with('customer','user')->find($id);
        $quotationList   = QuotationItem::where('quotation_id', $id)->get();

        return view('inventory.quotation.print', ['quotation' => $quotation, 'quotationList' => $quotationList]);        
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $quotation       = Quotation::with('customer','user')->find($id);
        $quotationList   = QuotationItem::where('quotation_id', $id)->get();

        $customers = Customer::get();
        $customers = json_encode($this->__prepareCustomerData($customers));

        $statusArr = [Quotation::STATUS_DRAFT,Quotation::STATUS_ACCEPTED,Quotation::STATUS_REJECTED];
        
        return view('inventory.quotation.edit', ['statusArr' => $statusArr, 'customers' => $customers, 'quotation' => $quotation, 'quotationList' => $quotationList]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request)
    {      
        try 
        {    
            $input = $request->all();

            $quotation_id = $request['quotation_id'];

            \DB::transaction(function() use ($input, $quotation_id) {

                $quotation                      = Quotation::find($quotation_id);
                $quotation->user_id             = \Auth::id();
                $quotation->customer_id         = $input['customer_id'];
                $quotation->quotation_ref       = $input['quotation_ref'];
                $quotation->enquiry_ref         = $input['enquiry_ref'];
                $quotation->requested_by        = $input['requested_by'];
                $quotation->quotation_date      = date('Y-m-d',strtotime($input['quotation_date']));  
                $quotation->expiry_date         = date('Y-m-d');     
                $quotation->expiry_days         = 0; 
                $quotation->notes               = $input['notes'];
                $quotation->status              = Quotation::STATUS_DRAFT;            

                $quotation->sub_total           = $input['final_total_bill'];
                $quotation->total               = $input['grand_total'];
                $quotation->tax                 = $input['tax_percent'];

                $quotation->update();

                $quotation_product = array();

                QuotationItem::where('quotation_id', $quotation_id)->delete();

                $quotation_product = array();

                foreach ($input['item_name'] as $key => $value) {

                    $itemData = Product::where("id",$input['item_id'][$key])->first();
                    
                    $quotation_product[$key]['quotation_id'] = $quotation_id;
                    $quotation_product[$key]['name'] = $itemData->name;
                    $quotation_product[$key]['description'] = $itemData->description;
                    $quotation_product[$key]['unit'] = $input['unit'][$key];
                    $quotation_product[$key]['quantity'] = $input['item_quantity'][$key];
                    $quotation_product[$key]['price'] = $input['price'][$key];
                    $quotation_product[$key]['total'] = $input['sub_total'][$key];
                    $quotation_product[$key]['product_id'] = $input['item_id'][$key];
                }

                QuotationItem::insert($quotation_product);
            });
        
            $messageType = 1;
            $message = " created successfully !";
            $message = "Quotation updated successfully !";

        } 
        catch(\Illuminate\Database\QueryException $ex){  
            $messageType = 2;
            $message = "Quotation update failed !";                
            \Log::error('Quotation update failed: ' . $ex->getMessage());        
        }        

        if($messageType == 1)
        	return redirect(url("/quotation/view"))->with(['success' => $message] );
       	else
       		return redirect(url("/quotation/view"))->with(['error' => $message] );
    }

    public function updateStatus(Request $request)
    {
        try 
        {    
            $input = $request->all();

            $quotation_id = $request['quotation_id'];

            \DB::transaction(function() use ($input, $quotation_id) {
                $quotation              = Quotation::find($quotation_id);
                $quotation->user_id     = \Auth::id();
                $quotation->status      = $input['status'];
                $quotation->update();
            });
        
            $messageType = 1;
            $message = " created successfully !";
            $message = "Quotation status updated successfully !";

        } 
        catch(\Illuminate\Database\QueryException $ex){  
            $messageType = 2;
            $message = "Quotation status update failed !";                
            \Log::error('Quotation status update failed: ' . $ex->getMessage());        
        }      
        
        if($messageType == 1)
        	return redirect(url("/quotation/view"))->with(['success' => $message] );
       	else
       		return redirect(url("/quotation/view"))->with(['error' => $message] );

    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($purchase_id)
    {
        try {
            
            $purchase = StockMove::find($purchase_id);
            $stodet = Quotation::where('stock_id', $purchase->stock_id)->first();

            if($purchase->branch_id == 1) {

                Quotation::where('stock_id', $purchase->stock_id)
                    ->update([
                            'stock_quantity'    => $stodet->stock_quantity + $purchase->quantity,
                            'b1_stock_quantity' => $stodet->b1_stock_quantity - $purchase->quantity
                    ]); 
            } else {
                Quotation::where('stock_id', $purchase->stock_id)
                    ->update([
                            'stock_quantity' => $stodet->stock_quantity + $purchase->quantity,
                            'b2_stock_quantity' => $stodet->b2_stock_quantity - $purchase->quantity
                    ]); 
            } 

            $purchase->delete();    

            $messageType = 1;
            $message = "Purchase details deleted successfully !";

        } catch(\Illuminate\Database\QueryException $ex){  
            $messageType = 2;
            $message = "Purchase deletion failed !";
        }
        
        return redirect(url("/purchase/view"))->with('messageType',$messageType)->with('message',$message);
    } 
}
